jQuery(document).ready(function ($) {
    let nodes = new vis.DataSet();
    let edges = new vis.DataSet();
    let network = null;
    let originalEdges = [];

    function draw(response) {
        nodes.clear();
        edges.clear();
        originalEdges = response.edges;

        let linkCounts = {};
        response.edges.forEach(edge => {
            if (linkCounts[edge.to]) {
                linkCounts[edge.to]++;
            } else {
                linkCounts[edge.to] = 1;
            }
        });

        response.nodes.forEach(node => {
            let size = 20 + (linkCounts[node.id] ? linkCounts[node.id] * 5 : 0);
            nodes.add({
                id: node.id,
                label: $('#toggle-title-length').is(':checked') ? node.short_label : node.full_label,
                full_label: node.full_label,
                short_label: node.short_label,
                url: node.url,
                post_type: node.post_type,
                categories: node.categories,
                value: size,
                color: {
                    background: node.color,
                    border: '#000000'
                },
                font: {
                    color: node.font.color
                },
                fixed: false
            });
        });

        response.edges.forEach(edge => {
            edges.add({
                from: edge.from,
                to: edge.to,
                arrows: edge.arrows
            });
        });

        $('#category-filters').empty();
        Object.entries(response.categories_colors).forEach(([category_id, category_info]) => {
            $('#category-filters').append(
                `<label><input type="checkbox" class="category-filter" value="${category_info.name}" checked><span class="category-color" style="background-color:${category_info.color};"></span> ${category_info.name}</label> `
            );
        });

        $('#select-all-categories').click(function () {
            $('.category-filter').prop('checked', true).trigger('change');
        });

        $('#deselect-all-categories').click(function () {
            $('.category-filter').prop('checked', false).trigger('change');
        });

        const container = document.getElementById('mynetwork');
        const data = { nodes: nodes, edges: edges };
        const options = {
            physics: {
                enabled: true,
                repulsion: {
                    centralGravity: 0.1,
                    springLength: 250,
                    springConstant: 0.05,
                    nodeDistance: 250,
                    damping: 0.09
                },
                solver: 'repulsion'
            },
            nodes: {
                shape: 'box',
                scaling: {
                    min: 10,
                    max: 100,
                    label: {
                        min: 14,
                        max: 30,
                        drawThreshold: 1,
                        maxVisible: 20
                    }
                },
                font: {
                    size: 12,
                    face: 'arial',
                    color: '#343434'
                }
            },
            edges: {
                color: { color: '#000000' }
            }
        };
        network = new vis.Network(container, data, options);

        network.on("stabilizationProgress", function (params) {
            var widthFactor = params.iterations / params.total;
            var percent = Math.round(widthFactor * 100);

            document.getElementById('text').innerHTML = percent + '%';
        });

        network.once("stabilizationIterationsDone", function () {
            document.getElementById('text').innerHTML = '100%';
            document.getElementById('loadingBar').style.opacity = 0;
            setTimeout(function () { document.getElementById('loadingBar').style.display = 'none'; }, 500);
            network.setOptions({ physics: { enabled: false } });
        });

        $('#toggle-title-length').change(function () {
            const isChecked = $(this).is(':checked');
            nodes.update(nodes.get().map(node => ({
                id: node.id,
                label: isChecked ? node.short_label : node.full_label
            })));
        });

        $('#show-posts, #show-pages').change(function () {
            filterNodes();
        });

        $('.category-filter').change(function () {
            filterNodes();
        });

        function filterNodes() {
            const showPosts = $('#show-posts').is(':checked');
            const showPages = $('#show-pages').is(':checked');
            const selectedCategories = $('.category-filter:checked').map(function () {
                return $(this).val();
            }).get();

            const nodesToUpdate = nodes.get().filter(node => {
                const shouldBeHidden = (node.post_type === 'post' && !showPosts) ||
                                       (node.post_type === 'page' && !showPages) ||
                                       (node.categories.length > 0 && !node.categories.some(category => selectedCategories.includes(category)));
                return node.hidden !== shouldBeHidden;
            }).map(node => ({
                id: node.id,
                hidden: (node.post_type === 'post' && !showPosts) || 
                        (node.post_type === 'page' && !showPages) || 
                        (node.categories.length > 0 && !node.categories.some(category => selectedCategories.includes(category)))
            }));

            nodes.update(nodesToUpdate);
        }

        network.on('click', function (params) {
            if (params.nodes.length > 0) {
                let selectedNodeId = params.nodes[0];
                let connectedNodes = network.getConnectedNodes(selectedNodeId);
                let allNodes = nodes.get({ returnType: "Object" });

                nodes.update(Object.values(allNodes).map(node => ({
                    id: node.id,
                    hidden: !connectedNodes.includes(node.id) && node.id !== selectedNodeId
                })));

                let allEdges = edges.get({ returnType: "Object" });

                edges.update(Object.values(allEdges).map(edge => ({
                    id: edge.id,
                    hidden: !connectedNodes.includes(edge.to) && !connectedNodes.includes(edge.from)
                })));
            }
        });

        network.on('doubleClick', function (params) {
            if (params.nodes.length === 0) {
                filterNodes();
                edges.update(edges.get().map(edge => ({
                    id: edge.id,
                    hidden: false
                })));
            } else if (params.nodes.length === 1) {
                let node = nodes.get(params.nodes[0]);
                if (node && node.url) {
                    window.open(node.url, '_blank');
                }
            }
        });

        $('#no-inbound-links-title').text(`被リンクゼロのページ：${response.no_inbound_links.length}件`);
        $('#no-inbound-links-table tbody').empty();
        response.no_inbound_links.forEach(post => {
            $('#no-inbound-links-table tbody').append(
                `<tr>
                    <td>${post.id}</td>
                    <td>${post.title}</td>
                    <td><button onclick="window.open('${post.edit_url}', '_blank')">編集</button></td>
                    <td><button onclick="window.open('${post.view_url}', '_blank')">表示</button></td>
                </tr>`
            );
        });

        $('#no-outbound-links-title').text(`発リンクゼロのページ：${response.no_outbound_links.length}件`);
        $('#no-outbound-links-table tbody').empty();
        response.no_outbound_links.forEach(post => {
            $('#no-outbound-links-table tbody').append(
                `<tr>
                    <td>${post.id}</td>
                    <td>${post.title}</td>
                    <td><button onclick="window.open('${post.edit_url}', '_blank')">編集</button></td>
                    <td><button onclick="window.open('${post.view_url}', '_blank')">表示</button></td>
                </tr>`
            );
        });
    }

    $('#generate-link-map').on('click', function() {
        document.getElementById('loadingBar').style.display = 'block';
        document.getElementById('loadingBar').style.opacity = 1;
        document.getElementById('text').innerHTML = '0%';
        $.post(lmiData.ajax_url, { action: 'lmi_get_dataset' }, function(response) {
            draw(response);
        });
    });

    $('#search-button').on('click', function() {
        executeSearch();
    });

    $('#node-search').on('keypress', function(e) {
        if (e.which == 13) { // Enter key pressed
            executeSearch();
        }
    });

    $('#reset-search-button').on('click', function() {
        $('#node-search').val(''); // 検索窓をクリア
        let nodesToUpdate = nodes.get().map(node => ({
            id: node.id,
            color: {
                background: node.color.background,
                border: '#000000'
            },
            font: { color: node.font.color },
            borderWidth: 1,
            borderWidthSelected: 1
        }));
        nodes.update(nodesToUpdate);
    });

    function executeSearch() {
        let searchTerm = $('#node-search').val().toLowerCase();
        let searchTerms = searchTerm.split(/[\s\u3000]+/); // スペースと全角スペースで分割
        let nodesToUpdate = nodes.get().map(node => {
            let match = searchTerms.every(term => node.label.toLowerCase().includes(term)); // AND検索
            return {
                id: node.id,
                color: node.color,
                font: node.font,
                borderWidth: match ? 2 : 1,
                borderWidthSelected: match ? 2 : 1,
                color: {
                    background: node.color.background,
                    border: match ? 'red' : '#000000'
                }
            };
        });
        nodes.update(nodesToUpdate);
    }

    // 拡大・縮小ボタンのクリックイベントを処理
    $('#zoom-in').on('click', function () {
        zoomNetwork(1.2); // 1.2倍に拡大
    });

    $('#zoom-out').on('click', function () {
        zoomNetwork(0.8); // 0.8倍に縮小
    });

    function zoomNetwork(scale) {
        const scaleOptions = {
            scale: scale,
            animation: {
                duration: 300,
                easingFunction: "easeInOutQuad"
            }
        };
        const currentPosition = network.getViewPosition();
        network.moveTo({
            position: currentPosition,
            scale: network.getScale() * scaleOptions.scale,
            animation: scaleOptions.animation
        });
    }
});
